<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  J2Store.payment_payfast
 *
 * @copyright Copyright (C) 2018 J2Store. All rights reserved.
 * @copyright Copyright (C) 2024 J2Commerce, LLC. All rights reserved.
 * @license https://www.gnu.org/licenses/gpl-3.0.html GNU/GPLv3 or later
 * @website https://www.j2commerce.com
 */

defined('_JEXEC') or die('Restricted access');
use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;

require_once(JPATH_ADMINISTRATOR . '/components/com_j2store/library/plugins/payment.php');

class plgJ2StorePayment_payfast extends J2StorePaymentPlugin
{
    /**
     * @var $_element  string  Should always correspond with the plugin's filename,
     *                         forcing it to be unique
     */
    var $_element = 'payment_payfast';
    var $_isLog = false;

    /**
     * @param object $subject The object to observe
     * @param array $config An array that holds the plugin configuration
     */
    function __construct(&$subject, $config)
    {
        parent::__construct($subject, $config);
        $this->loadLanguage('', JPATH_ADMINISTRATOR);

        if ($this->params->get('debug', 0)) {
            $this->_isLog = true;
        }
    }
    function onJ2StoreIsJ2Store4($element){
        if (!$this->_isMe($element)) {
            return null;
        }
        return true;
    }
    /**
     *
     * @param $data array
     *            form post data
     * @return string HTML to display
     */
    function _prePayment($data)
    {

        // prepare the payment form
        $vars = new \stdclass ();
        $platform= J2Store::platform();
        $fof_helper = J2store::fof();
        $order = $fof_helper->loadTable('Order', 'J2StoreTable',array( 'order_id' => $data ['order_id']));
        $currency_values = $this->getCurrency($order, false);
        $invoice_number = $order->getInvoiceNumber();
        $rootURL = rtrim(Uri::base(), '/');
        $subpathURL = Uri::base(true);
        if (!empty ($subpathURL) && ($subpathURL != '/')) {
            $rootURL = substr($rootURL, 0, -1 * strlen($subpathURL));
        }

        // set payment plugin variables
        $passPhrase = $this->_getParam('passPhrase');
        $payfast_merchant_id = $this->_getParam('merchant_id');
        $payfast_merchant_key = $this->_getParam('merchant_key');
        if ($this->params->get('sandbox')) {
            $payfast_merchant_id = $this->_getParam('sandbox_id');
            $payfast_merchant_key = $this->_getParam('sandbox_key');
        }


        $orderinfo = $order->getOrderInformation();
        $currency = J2Store::currency();

        $varsArray = array(
            'merchant_id' => $payfast_merchant_id,
            'merchant_key' => $payfast_merchant_key,
            'return_url' => $rootURL . $platform->getCheckoutUrl(array('task' => 'confirmPayment','orderpayment_type' => $this->_element ,'paction'=>'display')),
            'cancel_url' => $rootURL .$platform->getCheckoutUrl(array('task'=>'confirmPayment','orderpayment_type'=>$this->_element,'paction'=>'cancel','order_id'=>$data ['order_id'])),
            'notify_url' => $rootURL .$platform->getCheckoutUrl(array('task'=>'confirmPayment','orderpayment_type'=>$this->_element ,'paction'=>'process','tmpl'=>'component')),
            'name_first' => $orderinfo->billing_first_name,
            'name_last' => $orderinfo->billing_last_name,
            'email_address' => $order->user_email,
            'm_payment_id' => $data ['order_id'],
            'amount' => round($currency->format($order->order_total, $currency_values['currency_code'], $currency_values['currency_value'], false), 2),
            'item_name' => Text::sprintf('J2STORE_PAYFAST_ORDERITEM_NAME', $invoice_number),
            'item_description' => $this->params->get('order_description', '')
        );

        $secureString = '';
        foreach ($varsArray as $k => $v) {
            $secureString .= $k . '=' . urlencode(trim($v)) . '&';
            $vars->$k = $v;
        }

        if (!empty ($passPhrase) && !$this->params->get('sandbox')) {
            $secureString .= 'passphrase=' . urlencode($passPhrase);
        } else {
            $secureString = substr($secureString, 0, -1);
        }

        $securityHash = md5($secureString);
        $vars->signature = $securityHash;

        $vars->currency_code = $currency_values ['currency_code'];
        $vars->orderpayment_type = $this->_element;
        $vars->display_name = $this->params->get('display_name', 'PAYFAST');
        $vars->onbeforepayment_text = $this->params->get('onbeforepayment', '');
        $vars->button_text = $this->params->get('button_text', 'J2STORE_PLACE_ORDER');
        $vars->post_url = $this->_getPostUrl();

        return $this->_getLayout('prepayment', $vars);
    }

    function onJ2StoreCalculateFees($order)
    {
        //is customer selected this method for payment ? If yes, apply the fees
        $payment_method = $order->get_payment_method();

        if ($payment_method == $this->_element) {
            $total = $order->order_subtotal + $order->order_shipping + $order->order_shipping_tax;
            $surcharge = 0;
            $surcharge_percent = $this->params->get('surcharge_percent', 0);
            $surcharge_fixed = $this->params->get('surcharge_fixed', 0);
            if ($total > 0 && ((float)$surcharge_percent > 0 || (float)$surcharge_fixed > 0)) {
                //percentage
                if ((float)$surcharge_percent > 0) {
                    $surcharge += ($total * (float)$surcharge_percent) / 100;
                }

                if ((float)$surcharge_fixed > 0) {
                    $surcharge += (float)$surcharge_fixed;
                }

                $name = $this->params->get('surcharge_name', Text::_('J2STORE_CART_SURCHARGE'));
                $tax_class_id = $this->params->get('surcharge_tax_class_id', '');
                $taxable = false;
                if ($tax_class_id && $tax_class_id > 0) $taxable = true;
                if ($surcharge > 0) {
                    $order->add_fee($name, round($surcharge, 2), $taxable, $tax_class_id);
                }

            }


        }

    }

    /**
     * Processes the payment form
     * and returns HTML to be displayed to the user
     * generally with a success/failed message
     *
     * @param $data     array       form post data
     * @return string   HTML to display
     */
    function _postPayment($data)
    {
        $platform = J2Store::platform();
        $app = $platform->application();
        // Process the payment
        $paction = $app->input->getString('paction');

        $vars = new \stdClass();

        switch ($paction) {
            case "display":
                $session = Factory::getApplication()->getSession();
                $session->set('j2store_cart', array());
                $vars->message = Text::_($this->params->get('onafterpayment', ''));
                $html = $this->_getLayout('message', $vars);
                $html .= $this->_displayArticle();

                break;
            case "process":
                $vars->message = $this->_process();
                $html = $this->_getLayout('message', $vars);
                echo $html; // TODO Remove this
                $app->close();
                break;
            case "cancel":
                $vars->message = Text::_($this->params->get('oncancelpayment', ''));
                $order_id = $app->input->getString('order_id');
                $order = J2Store::fof()->loadTable('Order', 'J2StoreTable',array('order_id' => $order_id));//
                if ($order_id == $order->order_id) {
                    $order->update_status(6);
                    $order->store();
                }
                $html = $this->_getLayout('message', $vars);
                break;
            default:
                $vars->message = Text::_($this->params->get('onerrorpayment', ''));
                $html = $this->_getLayout('message', $vars);
                break;
        }

        return $html;
    }

    /**
     * Prepares variables for the payment form
     *
     * @return unknown_type
     */
    function _renderForm($data)
    {
        $user = Factory::getApplication()->getIdentity();
        $vars = new \stdclass();
        $vars->onselection_text = $this->params->get('onselection', '');

        return $this->_getLayout('form', $vars);
    }

    /**
     * Gets the Paypal gateway URL
     *
     * @param boolean $full
     * @return string
     * @access protected
     */
    function _getPostUrl($full = true)
    {
        $url = $this->params->get('sandbox') ? 'sandbox.payfast.co.za' : 'www.payfast.co.za';
        if ($full) {
            $url = 'https://' . $url;
        }
        return $url;
    }


    /**
     * Gets the value for the Paypal variable
     *
     * @param string $name
     * @return string
     * @access protected
     */
    function _getParam($name, $default = '')
    {
        $return = $this->params->get($name, $default);

        $sandbox_param = "sandbox_$name";
        $sb_value = $this->params->get($sandbox_param);
        if ($this->params->get('sandbox') && !empty($sb_value)) {
            $return = $this->params->get($sandbox_param, $default);
        }

        return $return;
    }

    /**
     * Validates the IPN data
     *
     * @param array $data
     * @return string Empty string if data is valid and an error message otherwise
     * @access protected
     */
    function validateITN($data)
    {
        $errors = array();
        $pfParamString = '';
        $pfHost = 'https://' . $this->_getPostUrl(false) . '/eng/query/validate';

        //check step - 1
        //dumbing the submitted var and calculating security sign
        foreach ($data as $key => $val) {
            if ($key != 'signature') {
                $pfParamString .= $key . '=' . urlencode($val) . '&';
            }
        }
        $pfParamString = substr($pfParamString, 0, -1);
        $this->_log($pfParamString, 'ITN Post string');

        $pfTempParamString = $pfParamString;

        $pfPassPhrase = $this->_getParam('passPhrase');

        if (!empty($pfPassPhrase) && !($this->params->get('sandbox'))) {
            $pfTempParamString .= '&passphrase=' . urlencode($pfPassPhrase);
        }

        $signature = md5($pfTempParamString);

        if ($data['signature'] != $signature) {
            //$errors[] = Text::_('J2STORE_PAYFAST_SIGNATURE_MISMATCH');
        }

        //check step - 2 Verify source Ip
        $validHosts = array('www.payfast.co.za', 'sandbox.payfast.co.za', 'w1w.payfast.co.za', 'w2w.payfast.co.za');
        $validIps = array();
        foreach ($validHosts as $hostname) {
            $ips = gethostbynamel($hostname);
            if ($ips) {
                $validIps = array_merge($validIps, $ips);
            }
        }
        $validIps = array_unique($validIps);

        if (count($validIps)) {
            $this->_log(implode('/n', $validIps), 'Valid IPS');
            $this->_log($_SERVER['REMOTE_ADDR'], 'Remote Address');
            if (!in_array($_SERVER['REMOTE_ADDR'], $validIps)) {

                $errors[] = Text::_('J2STORE_PAYFAST_PAYMENT_BAD_SOURCE_IP_ERROR');
            }
        }

        //Check -3
        $userAgent = $_SERVER['HTTP_USER_AGENT'];
        //try curl
        if (function_exists('curl_init')) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_USERAGENT, $userAgent);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HEADER, false);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // uncomment this line if you get no gateway response. ###
            curl_setopt($ch, CURLOPT_URL, $pfHost);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $pfParamString);
            $response = curl_exec($ch);
            curl_close($ch);
        } else {
            //if not use the fopen
            $header = "POST /eng/query/validate HTTP/1.0\r\n";
            $header .= "Host: " . $this->_getPostUrl() . "\r\n";
            $header .= "User-Agent: " . $userAgent . "\r\n";
            $header .= "Content-Type: application/x-www-form-urlencoded\r\n";
            $header .= "Content-Length: " . strlen($pfParamString) . "\r\n\r\n";
            $socket = fsockopen('ssl://' . $this->_getPostUrl(false), 443, $errno, $errstr, 60);
            fputs($socket, $header, $pfParamString);
            $response = "";
            $headerDone = false;
            while (!feof($socket)) {
                $line = fgets($socket, 1024);
                if (strcmp($line, "\r\n" == 0)) {
                    $headerDone = true;
                } else if ($headerDone) {
                    $response .= $line;
                }
            }

            fclose($socket);
        }

        $lines = explode("\r\n", $response);
        $this->_log($response, 'Validation response');
        $verifyResult = trim($lines[0]);
        if (strcasecmp($verifyResult, 'VALID') != 0) {
            $errors[] = Text::_('J2STORE_PAYFAST_VALIDATION_FAILURE');
        }

        if (count($errors)) {

            //there are errors. Return them as string
            return implode('/n', $errors);
        }

        //if we are here, validation is successful
        return '';
    }

    /**
     *
     * @return HTML
     */
    function _process()
    {
        $platfrom = J2Store::platform();
        $app = $platfrom->application();
        $data = $app->input->getArray($_POST);
        $errors = array();

        // remove slashes
        foreach ($data as $key => $value) {
            $data [$key] = stripslashes($value);
        }

        // log the transaction details
        $transaction_details = $this->_getFormattedTransactionDetails($data);
        $this->_log($transaction_details, 'Payment response');
        // get the unique order id
        if (isset ($data ['m_payment_id']) && !empty ($data ['m_payment_id'])) {
            $orderpayment = J2Store::fof()->loadTable('Order', 'J2StoreTable',array('order_id' => $data ['m_payment_id']));
            if ($orderpayment->order_id == $data ['m_payment_id']) {

                // ok we have a valid order
                // validate the Payfast data
                $data ['int_validation_results'] = $this->validateITN($data);
                $this->_log($data ['int_validation_results'], 'Validation results');
                $transaction_details = $this->_getFormattedTransactionDetails($data);

                $orderpayment->transaction_id = $data ['pf_payment_id'];
                $orderpayment->transaction_details = $transaction_details;
                $orderpayment->transaction_status = $data ['payment_status'];

                if (empty ($data ['int_validation_results'])) {

                    // now go ahead processing the order
                    switch (strtoupper($data ['payment_status'])) {

                        case 'COMPLETE' :
                            $orderpayment->payment_complete();
                            //clear cart
                            $orderpayment->empty_cart();
                            break;

                        case 'PENDING' :
                            // set order to pending. Also notify the customer that it is pending
                            $orderpayment->update_status(4, true);
                            break;

                        case 'FAILED' :
                            $orderpayment->update_status(3);
                            break;
                        default :
                            $errors [] = Text::_('J2STORE_PAYFAST_INVALID_PAYMENT_STATUS');
                            break;
                    }
                } else {
                    $orderpayment->update_status(3);
                    // validation failed.
                    $errors [] = $data ['int_validation_results'];
                }

                // store the order
                if (!$orderpayment->store()) {
                    $errors [] = $orderpayment->getErrors();
                }
            } else {
                // order id not found
                $errors [] = Text::_('J2STORE_PAYFAST_ORDER_ID_NOT_FOUND');
            }
        } else {
            // no unique id found. So its an error
            $errors [] = Text::_('J2STORE_PAYFAST_ORDER_ID_NOT_FOUND');
        }

        if ($errors) {
            // send an emails to site's administrators with error messages
            $error = implode('/n', $errors);
            $config = Factory::getApplication()->getConfig();
            $sitename = $config->getValue('config.sitename');
            if (version_compare(JVERSION, '4.0', 'ge')) {
                $sitename = $config->get('sitename');
            }
            // send error notification to the administrators
            $subject = Text::sprintf('J2STORE_PAYFAST_EMAIL_PAYMENT_NOT_VALIDATED_SUBJECT', $sitename);

            $receivers = $this->_getAdmins();
            foreach ($receivers as $receiver) {
                $body = Text::sprintf('J2STORE_PAYFAST_EMAIL_PAYMENT_FAILED_BODY', $receiver->name, $sitename, Uri::root(), $error, $transaction_details);
                J2Store::email()->sendErrorEmails($receiver->email, $subject, $body);
            }
            return $error;
        }

        // if here, all went well
        $error = 'processed';
        return $error;
    }

    /**
     * Formatts the payment data for storing
     *
     * @param array $data
     * @return string
     */
    function _getFormattedTransactionDetails($data)
    {
        $separator = "\n";
        $formatted = array();

        foreach ($data as $key => $value) {
            if ($key != 'view' && $key != 'layout') {
                $formatted[] = $key . ' = ' . $value;
            }
        }

        return count($formatted) ? implode("\n", $formatted) : '';
    }
}
