/**
 * @package   	JCE
 * @copyright 	Copyright (c) 2009-2024 Ryan Demmer. All rights reserved.
 * @license   	GNU/GPL 2 or later - http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 * JCE is free software. This version may have been modified pursuant
 * to the GNU General Public License, and as distributed it includes or
 * is derivative of works licensed under the GNU General Public License or
 * other free or open source software licenses.
 */

/* global jQuery */

(function () {

    var tinymce = {
        settings: {
            language: "en",
            document_base_url: ""
        },

        i18n: {},

        /**
         * Adds a language pack
         *
         * @method addI18n
         * @param {String} p Prefix for the language items. For example en.myplugin
         * @param {Object} o Name/Value collection with items to add to the language group.
         * @source TinyMCE EditorManager.js
         * @copyright Copyright 2009, Moxiecode Systems AB
         * @licence GNU / LGPL 2 - http://www.gnu.org/copyleft/lesser.html
         *
         * Modified for JQuery
         */
        addI18n: function (p, o) {
            var i18n = this.i18n;

            function hasOwn(obj, prop) {
                return Object.prototype.hasOwnProperty.call(obj, prop);
            }

            function setProps(prefix, obj) {
                for (var key in obj) {
                    if (hasOwn(obj, key)) {
                        i18n[prefix + '.' + key] = obj[key];
                    }
                }
            }

            if (typeof p === 'string') {
                setProps(p, o);
            } else {
                for (var lc in p) {
                    if (hasOwn(p, lc)) {
                        var groupObj = p[lc];
                        for (var g in groupObj) {
                            if (hasOwn(groupObj, g)) {
                                var keyObj = groupObj[g];
                                for (var k in keyObj) {
                                    if (hasOwn(keyObj, k)) {
                                        var path = (g === 'common')
                                            ? lc + '.' + k
                                            : lc + '.' + g + '.' + k;
                                        i18n[path] = keyObj[k];
                                    }
                                }
                            }
                        }
                    }
                }
            }
        },

        getLang: function (n, dv) {
            return this.i18n[this.settings.language + '.' + n] || dv;
        },

        convertURL: function (src) {
            return src;
        },

        documentBaseURI: {
            toAbsolute: function (url) {
                if (/:\/\//.test(url)) {
                    return url;
                }

                // remove leading slash
                url = url.replace(/^\//, '');

                var root = tinymce.settings.document_base_url;

                // remove trailing slash
                root = root.replace(/\/$/, '');

                // join and return
                return root + '/' + url;
            },

            toRelative: function (url) {
                if (/:\/\//.test(url)) {
                    var base = tinymce.settings.document_base_url;
                    return url.replace(new RegExp('^' + base), '');
                }

                return url;
            }
        },

        windowManager: {
            params: {
                window: window.parent,
                callback: function (selected, data) {
                    var win = window.parent,
                        values = data[0],
                        url = values.url;

                    var elm = win.document.getElementById(tinymce.settings.element);

                    // Find closest .field-media-wrapper parent
                    var wrapper = elm;

                    while (wrapper && wrapper !== win.document) {
                        if (wrapper.classList && wrapper.classList.contains('field-media-wrapper')) {
                            break;
                        }
                        wrapper = wrapper.parentNode;
                    }

                    var inst = null;

                    if (wrapper) {
                        // Try dataset first
                        inst = wrapper.dataset && wrapper.dataset.fieldMedia;

                        // If dataset value is a reference to a global object, resolve it
                        if (inst && typeof win[inst] === 'object') {
                            inst = win[inst];
                        } else {
                            // Fall back to using the wrapper directly
                            inst = wrapper;
                        }
                    }

                    if (inst && typeof inst.setValue === 'function') {
                        return inst.setValue(url, values);
                    }

                    // Fallback: set value and trigger change
                    elm.value = url;

                    var event = win.document.createEvent('Event');
                    event.initEvent('change', true, true);
                    elm.dispatchEvent(event);
                }

            },

            // Open a modal window . This is only used from inside an existing modal, eg: Help or Image Editor
            open: function (args, params) {
                args.url += '&' + tinymce.settings.token + '=1&standalone=1';

                // Create wrapper div
                var div = document.createElement('div');
                div.className = 'uk-inline-help uk-modal';

                // Create HTML structure
                var html = ''
                    + '<div class="uk-modal-dialog uk-modal-dialog-blank">'
                    + '  <button class="uk-modal-close uk-close"></button>'
                    + '  <div class="uk-modal-header"><h3><strong>' + (args.title || '') + '</strong></h3></div>'
                    + '  <iframe src="' + args.url + '" frameborder="0"></iframe>'
                    + '</div>';

                // Set innerHTML
                div.innerHTML = html;

                // Handle close button click
                var closeBtn = div.querySelector('.uk-close');

                if (closeBtn) {
                    closeBtn.addEventListener('click', function (e) {
                        e.preventDefault();
                        div.classList.remove('uk-open');

                        // Delay removal to mimic animation delay
                        setTimeout(function () {
                            if (div.parentNode) {
                                div.parentNode.removeChild(div);
                            }
                        }, 150);
                    });
                }

                // Append to body and show
                document.body.appendChild(div);
                div.classList.add('uk-open');
                div.style.display = 'block';

                // Store params
                tinymce.windowManager.params = params;
            },

            // Close a modal window opened using a Joomla Modal method.
            close: function () {
                var win = window.parent;
            
                // Joomla 4+ modal
                if (win.Joomla && win.Joomla.Modal && typeof win.Joomla.Modal.getCurrent === 'function') {
                    var current = win.Joomla.Modal.getCurrent();
                    if (current && typeof current.close === 'function') {
                        return current.close();
                    }
                }
            
                // Legacy modal handlers
                if (typeof win.jModalClose === 'function') {
                    win.jModalClose();
                }
            
                if (win.SqueezeBox && typeof win.SqueezeBox.close === 'function') {
                    win.SqueezeBox.close();
                }
            
                // Joomla 3.5.x / 4.x Media Field modal fallback
                var modals = win.document.querySelectorAll('.modal.in [data-dismiss]');

                for (var i = 0; i < modals.length; i++) {
                    var event = win.document.createEvent('Event');
                    event.initEvent('click', true, true);
                    modals[i].dispatchEvent(event);
                }
            }
            
        },

        getParam: function (n, dv) {
            var v = this.settings[n];
            return (typeof v !== "undefined") ? v : dv;
        }
    };

    window.tinyMCE = window.tinymce = tinymce;
})();